/*
 * AP(r) Computer Science GridWorld Case Study:
 * Copyright(c) 2002-2006 College Entrance Examination Board 
 * (http://www.collegeboard.com).
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
import info.gridworld.grid.Grid;
import info.gridworld.grid.BoundedGrid;
import info.gridworld.grid.Location;
import info.gridworld.world.World;

import java.util.ArrayList;

/**
 * The <code>GameOfFifteen</code> class is the main application.
 * <br />
 * This class is not tested on the AP CS A and AB exams.
 * <p> 
 * copyright&copy; 2007 Dave Wittry (http://apcomputerscience.com)
 * @author Dave Wittry
 */
public class GameOfFifteen extends World<ColorTextPiece>
{   
   private boolean winner;
   
   /**
     * Constructs a bounded 4-by-4 grid
     */   
   public GameOfFifteen()
   {
     setGrid( new BoundedGrid<ColorTextPiece>(4, 4) );
     loadBoard();
     winner = false;
     setMessage("Rearrange the tiles in alphabetical row-major order.\n" +
                "Type the letter you want to move.");
   }
   
   /**
     * Loads the board/grid with the first 15 letters of the alphabet scrambled
     */   
   private void loadBoard() 
   {
     String[] letters = {"A","B","C","D","E","F","G","H","I","J","K","L","M","N","O"};
     ArrayList<String> list = new ArrayList<String>();
     for (String letter: letters)
       list.add(letter);
     
     int row=0, col=0;
     for (String letter: list) 
     {
       add( new Location(row, col), new ColorTextPiece(letter) );
       col++;
       if ( col > 3 ) 
       {
         col = 0;  
         row++;
       }
     }
     
     /* For shuffling, we have to be careful. We can't just
      * scramble any way we wish because there is the "15-14"
      * problem - read a website for explanation. In short, not all
      * random chosen starting situations are solvable. So, my scramble
      * scrambles the way someone would scramble the board that
      * you would buy in a store - since the pieces can't be removed,
      * if we start with a solved puzzle and make a whole bunch of
      * random legal moves (don't remove pieces and snap them back in),
      * then we'll still have a solvable puzzle when we're done.
      */ 
     for (int num=1; num<=200; num++) {
    	 makeALegalMove(getGrid());
     }
   }
   
   /**
    * The method is called by loadBoard in order to help it scramble
    * the board.
    */
   private void makeALegalMove(Grid<ColorTextPiece> gr) {
	   Location locOfOpenCell = null;
	   for (int row=0; row<gr.getNumRows(); row++)
		   for (int col=0; col<gr.getNumCols(); col++)
			   if ( gr.get(new Location(row,col)) == null)
				   locOfOpenCell = new Location(row,col);
	   ArrayList<Location> legalMoves = new ArrayList<Location>();
	   Location north = locOfOpenCell.getAdjacentLocation(Location.NORTH);
	   Location south = locOfOpenCell.getAdjacentLocation(Location.SOUTH);
	   Location east = locOfOpenCell.getAdjacentLocation(Location.EAST);
	   Location west = locOfOpenCell.getAdjacentLocation(Location.WEST);
	   if ( gr.isValid(north) )
		   legalMoves.add(north);
	   if ( gr.isValid(south) )
		   legalMoves.add(south);
	   if ( gr.isValid(east) )
		   legalMoves.add(east);
	   if ( gr.isValid(west) )
		   legalMoves.add(west);
	   int randomIndex = (int)(Math.random() * legalMoves.size());
	   gr.put(locOfOpenCell, gr.get(legalMoves.get(randomIndex)));
	   gr.remove(legalMoves.get(randomIndex));
   }
   
   /**
     * The method is called by the GUI when a keyboard event takes place
     * @param desc the String describing the key
     * @param loc the selected location in the grid at the time the key was pressed 
     * @return <code>true</code> if the world consumes the key press, <code>false</code
     * if the GUI should consume it<br>
     * @see <code>World</code> class
     */   
   public boolean keyPressed(String desc, Location loc) 
   {
     if ( winner ) 
       return true; // game over; no more play
     if ( !("A".compareTo(desc)<=0 && "Z".compareTo(desc)>=0) ) // not a legal keyboard entry
       return true;
     Grid<ColorTextPiece> grid = getGrid();
        
     ArrayList<Location> nbrs = grid.getOccupiedLocations();
     for (Location possLoc: nbrs)
       if ( grid.get(possLoc).getText().equals(desc) ) 
       {
         ArrayList<Location> emptyCells = grid.getEmptyAdjacentLocations(possLoc);
         if ( emptyCells.size() == 0 ) // no open spot next to that letter
           return true;    
         int dirToward = possLoc.getDirectionToward(emptyCells.get(0));
         if ( dirToward % 90 != 0)
           return true;    // can only move horizontally/vertically
         // *** see comment below for other options of this one stmt    
         add(emptyCells.get(0), remove(possLoc));
         winner = determineWinner();
         if ( winner )
           setMessage("You WIN");
         else
           setMessage("You moved letter " + desc + " from position " + possLoc);
         return true;
       }
       return true;
   }
/* Here are some code-equivalent options 
   to the statement just above where you see ***

1)
Grid<ColorTextPiece> grid = getGrid(); 
grid.put(emptyCells.get(0), grid.get(possLoc));
grid.remove(possLoc);

2)
Grid<ColorTextPiece> grid = getGrid();
add(emptyCells.get(0), grid.get(possLoc));
grid.remove(possLoc);

3)
Grid<ColorTextPiece> grid = getGrid();
add(emptyCells.get(0), grid.get(possLoc));
remove(possLoc);

4)
add(emptyCells.get(0), remove(possLoc));

NOTE: In general, when dealing with objects of type Actor in GridWorld, one should
always call removeSelfFromGrid() - and not do as done here by using Grid
methods to remove the object. The case study narrative makes a point of this to remind you.
*/

   /**
     * The method determines if there is a winning situation
     * @return <code>true</code> if there is a winning situation, <code>false</code> otherwise
     */   
   private boolean determineWinner()
   {
     Grid<ColorTextPiece> grid = getGrid();
     ArrayList<ColorTextPiece> list = new ArrayList<ColorTextPiece>();
     for (int row=0; row<grid.getNumRows(); row++) 
       for (int col=0; col<grid.getNumCols(); col++)
         list.add( grid.get(new Location(row,col)) );         

     for (int i=0; i<list.size()-2; i++) // don't use last location in array - it might be null
     {
       ColorTextPiece first = list.get(i);
       ColorTextPiece second = list.get(i+1);
       if ( first == null || second == null )
         return false; // one of them is the empty cell
       if ( first.getText().compareTo(second.getText()) > 0 )
         return false;
     }
     return true;
   }

    /**
     * The method is called by the GUI when a mouse event takes place. There are no mouse
     * events for this game - so this method immediately returns true;
     * @param loc the selected location in the grid at the time the mouse was pressed 
     * @return <code>true</code> if the world consumes the key press, <code>false</code
     * if the GUI should consume it<br>
     * @see <code>World</code> class
     */   
   public boolean locationClicked(Location loc)
   {
     return true; // do nothing when mouse-clicked
   }
   
   public static void main(String[] args)
   {
     World<ColorTextPiece> mw = new GameOfFifteen();
     System.setProperty("info.gridworld.gui.selection", "hide");
     mw.show(); 
   }
}