/*
 * AP(r) Computer Science GridWorld Case Study:
 * Copyright(c) 2002-2006 College Entrance Examination Board 
 * (http://www.collegeboard.com).
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
import info.gridworld.world.World;
import info.gridworld.grid.Grid;
import info.gridworld.grid.BoundedGrid;
import info.gridworld.grid.Location;

/**
 * The <code>WuZiQiGameRunner</code> is the main application. 
 * <br />
 * This class is not tested on the AP CS A and AB exams.
 * <p> 
 * copyright&copy; 2007 Dave Wittry (http://apcomputerscience.com)
 * @author Dave Wittry
 */
public class WuZiQiGameRunner extends World<GamePiece>
{
    private boolean isBlacksTurn;
    private boolean winner;

    /**
     * Construct a WuZiQi Game
     */
    public WuZiQiGameRunner() 
    {
      isBlacksTurn = true;
      winner = false;
      makeBoard();
      setMessage("WuZiQi(Gomoku) - Black Goes First");
    }

    /**
     * Initialize the board. It is initially just filled with all gray background
     * tiles (GamePiece.gif) so the board is beautiful.
     *
     */
    private void makeBoard() 
    {
      setGrid( new BoundedGrid<GamePiece>(9, 9) ); // recommended beginner gameboard size
      for (int r=0; r<getGrid().getNumRows(); r++)
        for (int c=0; c<getGrid().getNumCols(); c++)
          add(new Location(r,c), new GamePiece("")); // background tiles
    }
    
    /**
     * Determines the winner. Currently written in O(1), independent of N, 
     * the number of pieces currently played
     * @param loc loacation of the last piece just played
     * @return <code>true</code> if the piece just played at position <code>loc</code> completes
     * a win for the player who just moved, false if no such winner yet
     */
    private boolean determineWinner(Location loc) {
      // only need to check if 5 in a row from the current loc (last move made)
      int dir = Location.AHEAD, consecutive = 0;
      Location nextLoc;
      Grid<GamePiece> gr = getGrid();
      /* 
       * There are 4 main axes to check for 5-in-a-row.
       * This will give students excellent practice with the Location class.
       * see if they can come up with it on their own - reveal it to those who can't -
       * let each of them try to write it in a clear manner.
       */
      GamePiece piecePlayed = gr.get(loc);
      for (int numDiagonals=1; numDiagonals<=4; numDiagonals++) 
      {
        nextLoc = loc.getAdjacentLocation(dir);
        consecutive = 1; // current piece played counts as 1 so far
        for (int i=1; i<=2; i++) 
        {
          while ( gr.isValid(nextLoc) && gr.get(nextLoc).equals(piecePlayed) ) 
          {
            consecutive++;
            nextLoc = nextLoc.getAdjacentLocation(dir);
          }
          dir += Location.HALF_CIRCLE;
          nextLoc = loc.getAdjacentLocation(dir);
        }
        if ( consecutive >= 5 ) // winner (important to check for greater than as well)
          return true;
        dir += Location.HALF_CIRCLE;  // get back to the starting diagonal direction
        dir += Location.HALF_RIGHT;   // turn to the next diagonal
      }
      return false;
    }
    
    /**
     * When the current player clicks and attempts to place a piece, this method
     * see whether it is a legal move and, if so, places a piece and calls determineWinner
     * @param loc location passed from the GUI where the use just clicked
     */
    public boolean locationClicked(Location loc)
    {
      if ( winner )
        return true; 
      if ( getGrid().get(loc).isBackground() ) // player making a legal move
      {
        if ( isBlacksTurn ) 
        {
          add(loc, new GamePiece("_black"));
          setMessage("White's Turn - Black moved to " + loc);
        }
        else
        {
          add(loc, new GamePiece("_white"));
          setMessage("Black's Turn - White moved to " + loc);
        }
        if ( determineWinner(loc) ) 
        {
          winner = true;
          if ( isBlacksTurn )
            setMessage("Black WINS");
          else
            setMessage("White WINS");
        }
        isBlacksTurn = !isBlacksTurn;
      }
      else // player making an illegal move
      {
        if ( isBlacksTurn )
          setMessage("Occupied Cell - Black, it's still your turn");
        else
          setMessage("Occupied Cell - White, it's still your turn");
      }
      return true;      
    }
   
    public static void main(String[] args)
    {
      World wzq = new WuZiQiGameRunner();
      System.setProperty("info.gridworld.gui.selection", "hide"); // get rid of focus indicator on the gui
      wzq.show();
    }
}