/*
 * AP(r) Computer Science GridWorld Case Study:
 * Copyright(c) 2002-2006 College Entrance Examination Board 
 * (http://www.collegeboard.com).
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
import info.gridworld.grid.Location;
import info.gridworld.world.World;

import java.util.List;
import java.util.ArrayList;
import java.util.Scanner;
import java.util.StringTokenizer;

import java.awt.Color;
import java.io.*;
/**
 * The <code>TrafficJam</code> class is the main application.
 * <br />
 * This class is not tested on the AP CS A and AB exams.
 * <p> 
 * copyright&copy; 2007 Dave Wittry (http://apcomputerscience.com)
 * @author Dave Wittry
 */
public class TrafficJam extends World<Piece>
{   
   private MultiCellBoundedGrid multiGrid;
   private ArrayList<ArrayList<Block>> gameChoices;
   private ArrayList<Integer> maxMovesPerGame;
   private int totalMovesForTheWin;
   private int gameRequested;
   private boolean waitingForFirstClick;
   private boolean winner;
   private boolean beginGame;
   private Block selectedBlock;
   private Block redBlock;
   private TimerFrame timerFrame;
   
   public TrafficJam()
   { 
      multiGrid = new MultiCellBoundedGrid(6, 6);
      gameChoices = new ArrayList<ArrayList<Block>>();
      maxMovesPerGame = new ArrayList<Integer>();
      totalMovesForTheWin = 0;
      waitingForFirstClick = true;
      winner = false;
      beginGame = false;
      selectedBlock = null;
      timerFrame = null;
          
      readVehicleData();
      setGrid( multiGrid );

      char last = (char)('A' + gameChoices.size() - 1);
      keyPressed("A", new Location(0,0)); //get the grid primed/displayed - normally, keyPressed is called by GUI
      String msg = "GOAL: Get red arrow to right wall." + 
                   "\nUse letters 'A' through '" + last + "' to cycle through " + gameChoices.size() + " starting situations. " +
                   "Press ENTER to begin.";
      setMessage(msg);
   }
   
   private void readVehicleData() {
       // note: data is found on many sites - mine from http://home.hetnet.nl/~fredvonk/RushHour.htm
       Scanner dataFile = null;
       int row=0, col=0, numCells=0;
       String orientation, line;
       Block blkInfo;
       ArrayList<Block> oneGameSituation;
       List<Color> colors;
       StringTokenizer st;
       
       try { 
         dataFile = new Scanner( new File("vehicleData.txt") );
       }
       catch (FileNotFoundException ex) 
       {
         System.out.println("** Can't find the file **");
         System.exit(1);
       }
       
       do   // read all game data - there are many games in the one data file
       {
         oneGameSituation = new ArrayList<Block>();
         colors =  new ArrayList<Color>();
         maxMovesPerGame.add( Integer.parseInt(dataFile.nextLine()) );
         loadColors(colors);
         line = dataFile.nextLine();
         while ( ! ("---".equals(line) || "xxx".equals(line)) ) // read one game situation
         {
           st = new StringTokenizer(line);
           row = Integer.parseInt(st.nextToken());
           col = Integer.parseInt(st.nextToken());
           orientation = st.nextToken();
           numCells = Integer.parseInt(st.nextToken());
           blkInfo = new Block(new Location(row, col), numCells, orientation, colors.remove(0));
           oneGameSituation.add(blkInfo);
           line = dataFile.nextLine();
         }
         gameChoices.add(oneGameSituation);
       } while ( !"xxx".equals(line) ); 
       dataFile.close();
   }
   
   public void loadColors(List<Color> colors) 
   {
     // there are 17 colors here - if your game has more than 17 blocks, you'll need to add colors
     // red should be first - others don't matter - red is the main block's color(piece you win with) 
     // the data file is set up so that the position of the red car always comes first
     colors.add(Color.RED); 

     colors.add(Color.CYAN);   colors.add(Color.BLUE);    colors.add(Color.GREEN); colors.add(Color.YELLOW);
     colors.add(Color.ORANGE); colors.add(Color.MAGENTA); colors.add(Color.PINK);  colors.add(Color.GRAY);
     colors.add(Color.BLACK); 
        
     colors.add(Color.YELLOW.darker());   colors.add(Color.CYAN.darker());  colors.add(Color.BLUE.darker().darker());
     colors.add(Color.ORANGE.darker());   colors.add(Color.GREEN.darker());
     colors.add(Color.MAGENTA.darker());  colors.add(Color.PINK.darker());        
   }
   
   public boolean keyPressed(String desc, Location loc) 
   {
     if ( beginGame ) 
       return true; // game has already started, disable keyboard
     if ( "ENTER".equalsIgnoreCase(desc) ) 
     {
       beginGame = true;
       setMessage("Click an arrow and then an open, legal cell to move it.");
       return true;
     }
     if ( !("A".compareTo(desc) <= 0 && "Z".compareTo(desc)>=0) ) // not a legal keyboard entry
       return true;
     gameRequested = desc.codePointAt(0) - 65; // "A"=65
     if ( gameRequested+1 > gameChoices.size() ) // illegal choice
       return true;
     ArrayList<Block> game = gameChoices.get(gameRequested);
     redBlock = game.get(0); // first block is always the red one
     multiGrid.clearGrid();
 
     for (Block blk: game)
       multiGrid.putBlock(blk.getLocation(), blk);
     String msg = "Game " + desc + " chosen. It can be done in " + maxMovesPerGame.get(gameRequested) + " moves." +
                  "\nPress ENTER to begin play or another letter for a different game.";
     setMessage(msg);
     return true;
   }

   public boolean locationClicked(Location loc)
   {
     if ( !beginGame || winner ) // haven't chosen game yet or the game is already over
       return true; 
     if ( timerFrame == null ) // turn timer on 
     {
       timerFrame = new TimerFrame();
       timerFrame.startTime();
     }
     if ( waitingForFirstClick ) 
     { 
       Block tmpBlk = (Block)multiGrid.findBlock(loc);
       if ( tmpBlk != null ) // they clicked on a block
       {
         setMessage("" + tmpBlk.getLocation());
         selectedBlock = tmpBlk;
         waitingForFirstClick = false;
         setMessage(tmpBlk + " Block was clicked. Click again to move it.");
       } // else, they didn't click on a block; still waitingForFirstClick
     }
     else // second time they've clicked
     {
       if ( multiGrid.findBlock(loc) == null &&
            multiGrid.findLegalMovingLoc(selectedBlock, loc) != null ) // position is open and in same row/col
       {     
         winner = moveAndUpdateBlock(selectedBlock, loc);
         if ( winner ) 
         {
           timerFrame.stopTime();
           String msg = "You WON in " + (totalMovesForTheWin+1) + " moves! Best possible:" + 
                        maxMovesPerGame.get(gameRequested) + " Game Played:" + (char)(gameRequested+'A');
           setMessage(msg);
           return true;
         }
         totalMovesForTheWin++;
         setMessage("" + totalMovesForTheWin + " moves so far. Best possible:" + maxMovesPerGame.get(gameRequested));
         waitingForFirstClick = true;
         selectedBlock = null;
       }
       else // they clicked on another block or empty cell
       {
         if ( multiGrid.findBlock(loc) != null ) //another block - make that one the one they now want to move
         {
           waitingForFirstClick = false;
           selectedBlock = (Block)multiGrid.findBlock(loc);
           setMessage(selectedBlock + " Block was clicked. Click again to move it.");
         }
       }
     }       
     return true;      
   }

    /*
    * pre: loc is valid, available(empty), and legal for Block to move to
    * The cells in grid will be updated - blk will also be updated.
    * Return true if winner - false, otherwise
    */
    public boolean moveAndUpdateBlock(Block blk, Location loc) 
    {
      Location legalLoc = multiGrid.findLegalMovingLoc(blk, loc);
        
      // legal move, so clear out old cells from grid, find new blk position, 
      // fill grid again, and then check for winner
      Piece[] oldCells = blk.getCells();
      blk.updatePosition(legalLoc);
      Piece[] newCells = blk.getCells();
      for (Piece p: oldCells)
        multiGrid.remove(p.getLocation());
      for (Piece p: newCells)
        multiGrid.put(p.getLocation(), p);
      int redBlockRightSide = redBlock.getLocation().getCol() + redBlock.getNumCells() - 1;
      if ( redBlockRightSide == 5 )
        return true; // winner
      else
        return false;
    }
    
   public static void main(String[] args)
   {
     World mw = new TrafficJam();
     System.setProperty("info.gridworld.gui.selection", "hide");
     mw.show();
   }
}