/*
 * AP(r) Computer Science GridWorld Case Study:
 * Copyright(c) 2002-2006 College Entrance Examination Board 
 * (http://www.collegeboard.com).
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

import java.util.Map;
import java.util.HashMap;

import info.gridworld.grid.BoundedGrid;
import info.gridworld.grid.Location;
/**
 * A <code>MultiCellBoundedGrid</code> is a rectangular grid with a finite number of
 * rows and columns. You can store objects in it which take up multiple Grid
 * locations.
 * <br />
 * This class is not tested on the AP CS A and AB exams.
 * <p> 
 * copyright&copy; 2007 Dave Wittry (http://apcomputerscience.com)
 * @author Dave Wittry
 */
public class MultiCellBoundedGrid extends BoundedGrid<Piece>
{
    private Map<Location, Blockable> blocks; // location will either be top of block(vertical), or left(horizontal)

    /**
     * Constructs a BoundedGrid
     * @param rows number of rows the Grid will contain
     * @param cols number of columns the Grid will contain
     */        
    public MultiCellBoundedGrid(int rows, int cols)
    {
      super(rows, cols);
      blocks = new HashMap<Location, Blockable>();
    }

    /**
     * Finds a particular <code>Block</code> within the Grid 
     * @param loc candidate location where a Block might lie
     * @return returns the particular <code>Block</code> for which <code>loc</code> lies within,
     * otherwise returns null if no such Block exists
     */            
    public Blockable findBlock(Location loc) 
    {
      for (Location l: blocks.keySet()) 
      {
        Blockable tmpBlk = blocks.get(l);
        if ( tmpBlk.contains(loc) != null )
          return tmpBlk;
      }
      return null;
    }

    /**
     * Determine if a given <code>Block</code> can be moved to a given location
     * @param blk <code>Block</code> to be moved
     * @param loc potential legal location for the move
     * @return If legal move, returns the location that the block's left or upper most part
     * would be moved to. Otherwise, illegal move so return false
     */            
    public Location findLegalMovingLoc(Blockable blk, Location loc) 
    {
      // doesn't need to move to self - so do nothing if they clicked on same block
      if ( blk.contains(loc) != null ) 
        return null; 
        
      boolean correctColumn = blk.isVertical() && loc.getCol() == blk.getLocation().getCol();
      boolean correctRow = !blk.isVertical() && loc.getRow() == blk.getLocation().getRow();
      if ( !( correctColumn || correctRow ) ) // illegal move for Block
        return null;

      int top=0, bottom=0, left=0, right=0, startCheck=0, stopCheck=0;  
      boolean clickedAbove = false, clickedLeft = false;
      if ( blk.isVertical() ) 
      {
        top = blk.getLocation().getRow();
        bottom = top + blk.getNumCells() - 1;
        if ( loc.getRow() < top ) // clicked above it
        {
          startCheck = loc.getRow();
          stopCheck = top - 1;
          clickedAbove = true;
        }
        else // clicked below it
        {
          startCheck = bottom + 1;
          stopCheck = loc.getRow();
        }
        if ( ! legalMovLocation(loc, blk.isVertical(), startCheck, stopCheck) ) 
        	return null;
        if ( clickedAbove )
          return loc;
        else
          return new Location ( top+(stopCheck-bottom), loc.getCol() );
      }
      else // it's a horizontal block
      {
        left = blk.getLocation().getCol();
        right = left + blk.getNumCells() - 1;
        if ( loc.getCol() < left ) // clicked to the left of it
        {
          startCheck = left;
          stopCheck = blk.getLocation().getCol() - 1;
          clickedLeft = true;
        }
        else // clicked to the right of it
        {
          startCheck = right + 1;
          stopCheck = loc.getCol();
        }
        if ( ! legalMovLocation(loc, blk.isVertical(), startCheck, stopCheck) )
        	return null;
        if ( clickedLeft )
          return loc;
        else
          return new Location ( loc.getRow(), left+(stopCheck-right) );
      }
    }

	private boolean legalMovLocation(Location loc, boolean isVertical, int startCheck, int stopCheck) {
		if ( isVertical ) {
		  for (int row=startCheck; row<=stopCheck; row++)
			if ( get(new Location(row, loc.getCol())) != null )
		      return false; // illegal position to move to
		}
		else { // horizontal
		  for (int col=startCheck; col<=stopCheck; col++)
		    if ( get(new Location(loc.getRow(), col)) != null )
		      return false; // illegal position to move to
		}
		return true;
	}

    /**
     * Places the individual <code>Piece</code>s of a <code>Block</code> into the Grid
     * @param loc location where the <code>Block</code> should be placed
     * @param blk <code>Block</code> which is being placed
     * @return returns the Block which was replaced
     */                
    public Blockable putBlock(Location loc, Blockable blk) 
    {
      Blockable oldBlk = blocks.remove(loc);
      blocks.put(loc, blk);
      // now put individual parts/pieces of Block into BoundedGrid
      for (Piece p: blk.getCells())
        if ( isValid(p.getLocation()) )
          put(p.getLocation(), p);
        else
          throw new IllegalArgumentException("illegal Block location being put in the MultiCell Grid");
      return oldBlk;
    }

    /**
     * Remove all <code>Block</code>s from the Grid
     */                
    public void clearGrid() 
    {
      for (Location loc: getOccupiedLocations())
        remove(loc);
      blocks = new HashMap<Location, Blockable>();
    }
}